// PCHotwire - Tragflächen-Schneidegerät
// Version 1.2
// Datum 29.Dec.11
// Versionskommentar: Alle Features komplett und getestet;
//                    HOME-Funktion implementiert
// ----------------------------------------------------------------------------------
// Interface für den PC/Linux
// Gekoppelt über serielle Schnittstelle mit Arduino
//
// Befehle, die über die serielle Schnittstelle gesendet werden:
//  '5' - Links Y-Achse nach oben
//  '6' - Links Y-Achse nach unten
//  '7' - Links X-Achse nach vorne
//  '8' - Links X-Achse nach hinten
//  '1' - Rechts Y-Achse nach oben
//  '2' - Rechts Y-Achse nach unten
//  '3' - Rechts X-Achse nach vorne
//  '4' - Rechts X-Achse nach hinten
//  '0' - Stoppt die Bewegung aller Achsen
//  'S' - Startet den Schneidevorgang, ab dann fordert der Arduino die Daten an
//  "<DIR><hex>" - bewegt beide Seiten um <hex> Schritte
//                 <DIR> = 'U'  Aufwärts
//                 <DIR> = 'D'  Abwärts
//                 <DIR> = 'F'  Vorwärts
//                 <DIR> = 'B'  Rückwärts
//                 Beispiel: F00F3
//  "<sign><StepHex><sign><XaspectHex><sign><YaspectHex>"
//                 - Übergibt einen Vector an den Arduino. Der Arduino weiß, ob der Vector 
//                   für die rechte oder die linke Seite gilt, weil er den Vector 
//                   vorher entsprechend angefragt hat. 
//                   Die drei Vorzeichen geben die Richtung an.
//                   Die drei Werte sind unsigned Hex (4 Characters).
//                   StepHex - Anzahl der Schritte in X-Richtung
//                   XAspectHex - nach jeweils wieviel Clockticks ein X-Motorschritt gemacht wird
//                   YAspectHex - nach jeweils  wieviel Clockticks ein Y-Motorschritt gemacht wird 
//                   Beispiel: +024E+0041-003D
//  "#----#----#----" - Keine weiteren Vectoren mehr, d.h. Arduino hört auf danach zu fragen.
//
// Befehle, die empfangen werden:
// 'L' - Arduino fordert nächsten Vector für linke Seite an
// 'R' - Arduino fordert nächsten Vector für rechte Seite an
// 'A' - Rechter Endschalter in X-Richtung erreicht
// 'B' - Linker Endschalter in X-Richtung erreicht
// 'a' - Rechter Endschalter in Y-Richtung erreicht
// 'b' - Linker Endschalter in Y-Richtung erreicht

import controlP5.*;
import processing.serial.*;

// Konstanten der Mechanik der Maschine:
final int STEPS_PER_MM = 48;  // Anzahl der Schritte pro Millimeter Verfahrweg
final float MOTOR_DIST = 850;  // Abstand der beiden X/Y-Führungen voneinander in Millimetern
final int X_STOP = 7;       // Abstand des Endschalters vom Nullpunkt des Styropors in Millimetern
final int Y_STOP = 5;       // Abstand des Endschalters von der Grundplatte in Millimetern

String[] configLines;
String configPath, configSerial;

String FoilLeft, FoilRight;  
String[] profilLinks, profilRechts;

// vier Arrays enthalten die aufbereiteten Schneidekoordinaten für Arduino; 
// Tiefe, Anstellwinkel etc. sind darin bereits berücksichtigt:
float[] plx, ply, prx, pry;   

// vier Arrays mit rohen Koordinaten aus Profildateien:
float[] plxRaw, plyRaw, prxRaw, pryRaw;

int idxLeft, idxRight; // Zähler für die Arrays mit den Profilkoordinaten
int Heizleistung = 70;

float lastXR = 0, lastYR = 0, lastXL = 0, lastYL = 0; // Koordinaten aus dem vorigen Schritt

float mittlereTiefe;

// wahre Tiefe an den Motorpositionen (entspricht bei rechteckigen Flächen programmierter Tiefe):
float tiefeL0, tiefeR0;   

String[] fileList;

// GUI:

ControlP5 controlP5;

Textfield TLinks;
Textfield TRechts;
Textfield AWLinks;
Textfield AWRechts;
Textfield Pfeilung;
Textfield SegLaenge;
Textfield xStart;
Textfield yStart;
Textfield Beplankung;

Textlabel ProfNameL;
Textlabel ProfNameR;

ListBox listL, listR;

Serial myPort; // serielle Schnittstelle

void setup() {// wird beim Programmstart einmalig aufgerufen
  int i;
  size(600, 520);
  frameRate(25);

  // Profildateien aus dem Unterverzeichnis profile_dat lesen
  File dataFolder = new File(sketchPath+"/profile_dat");
  fileList = dataFolder.list();
  fileList = sort(fileList);

  // GUI initialisieren:
  controlP5 = new ControlP5(this);

  listL = controlP5.addListBox("Profil Links", 30, 30, 120, 100);
  for (i=0; i<fileList.length; i++) listL.addItem(fileList[i], i);
  listR = controlP5.addListBox("Profil Rechts", 320, 30, 120, 100);
  for (i=0; i<fileList.length; i++) listR.addItem(fileList[i], i);

  TLinks = controlP5.addTextfield("Tiefe Links", 30, 140, 100, 20);
  TRechts = controlP5.addTextfield("Tiefe Rechts", 320, 140, 100, 20);
  AWLinks = controlP5.addTextfield("Anstellwinkel Links", 150, 140, 100, 20);
  AWLinks.setText("0");
  AWRechts = controlP5.addTextfield("Anstellwinkel Rechts", 440, 140, 100, 20);
  AWRechts.setText("0");
  Pfeilung = controlP5.addTextfield("Pfeilungswinkel", 190, 270, 80, 20);
  Pfeilung.setText("0");
  SegLaenge = controlP5.addTextfield("Segmentlaenge", 190, 320, 80, 20);
  xStart = controlP5.addTextfield("x-Start", 310, 270, 80, 20);
  xStart.setText("10");
  yStart = controlP5.addTextfield("y-Start", 310, 320, 80, 20);
  yStart.setText("20");
  Beplankung = controlP5.addTextfield("Beplankung", 190, 370, 80, 20);
  Beplankung.setText("0");
  controlP5.addSlider("Heizleistung", 0, 255, 70, 60, 435, 160, 14).setId(7);

  controlP5.addButton("Home", 0, 30, 480, 100, 19).setId(8);
  controlP5.addButton("Start", 0, 170, 480, 100, 19).setId(9);
  controlP5.addButton("Heiz-Test", 128, 380, 430, 100, 20).setId(12);
  controlP5.addButton("Cancel", 255, 310, 480, 100, 19).setId(10);
  controlP5.addButton("Exit", 128, 450, 480, 100, 19).setId(11);

  controlP5.addButton("UL", 128, 65, 280, 30, 30).setId(21);
  controlP5.addButton("FL", 128, 30, 320, 30, 30).setId(22);
  controlP5.addButton("BL", 128, 100, 320, 30, 30).setId(23);
  controlP5.addButton("DL", 128, 65, 360, 30, 30).setId(24);

  controlP5.addButton("UR", 128, 485, 280, 30, 30).setId(31);
  controlP5.addButton("FR", 128, 450, 320, 30, 30).setId(32);
  controlP5.addButton("BR", 128, 520, 320, 30, 30).setId(33);
  controlP5.addButton("DR", 128, 485, 360, 30, 30).setId(34);

  ProfNameL = controlP5.addTextlabel("labelL", "", 130, 210);
  ProfNameR = controlP5.addTextlabel("labelR", "", 420, 210);

  println("Serial Interfaces found!");
  println(Serial.list());

  // ini-Datei auslesen: 
  configLines = loadStrings("PCHotwire.ini");
  configSerial = configLines[0];

  println("Config File found!");
  println("Serial Interface: "+configSerial);
  println("sketchPath="+sketchPath);

  // die in der ini-Datei angegebene serielle Schnittstelle initialisieren:
  myPort = new Serial(this, configSerial, 9600);

  background(100);
  fill(100);
  stroke(60);
  rect(10, 10, 280, 240);
  rect(300, 10, 280, 240);  
  rect(10, 260, 150, 150);
  rect(170, 260, 250, 150);
  rect(430, 260, 150, 150);
  rect(10, 420, 570, 40);
}

// Die folgende Funktion wird automatisch 30 mal pro Sekunde aufgerufen, ausreichend oft, 
// um den Port abzufragen und keine Nachrichten vom Arduino zu verpassen.
// Der Arduino fordert jeweils rechts oder links den nächsten Vector an.

void draw() {
  char inByte='0';

  if (myPort.available() > 0) {    
    inByte = myPort.readChar();
    print(inByte);  // Debug-Ausgabe

      if (inByte=='L') {  // linke Seite braucht neuen Vector
      myPort.write(getNextVector('L'));  // Vector über serielle Schnittstelle senden
    }

    if (inByte=='R') {  // rechte Seite braucht neuen Vector
      myPort.write(getNextVector('R'));  // Vector über serielle Schnittstelle senden
    }
  }
}

//======================= Hilfsfunktionen ========================

// Berechnet den nächsten Vector, nach Seiten unterschieden

String getNextVector(char side) {
  float x, y, dx, dy;
  float stepsX, stepsY;
  float aspectX, aspectY;
  float tiefe;
  float speedFactor;
  int aspX, aspY, stp;
  String outStr, aspXhex, aspYhex, stpHex;
  char aspXsign, aspYsign, stpSign;
  int ende=0;
  int linDist;

  if (side=='L') {    // linke Seite
    tiefe = tiefeL0;  // wahre Tiefe an der linken Motorposition 
    x=plx[idxLeft];   // Koordinaten aus Arrays der aufbereiteten Profilkoordinaten lesen   
    y=ply[idxLeft];   
    dx=(x-lastXL);    // Differenz berechen zur vorigen Koordinate
    dy=(y-lastYL);
    lastXL=x;         // Koordinaten für den nächsten Durchgang kopieren
    lastYL=y;
    // falls noch nicht die letzte Koordinate erreicht ist, hochzählen
    if (idxLeft < (plx.length-1)) idxLeft++; 
    else ende=1; // ansonsten abbrechen
  } 
  else {            // rechte Seite analog zur linken 
    tiefe = tiefeR0;  
    x=prx[idxRight];  
    y=pry[idxRight];
    dx=(x-lastXR);
    dy=(y-lastYR);
    lastXR=x;
    lastYR=y;
    if (idxRight < prx.length-1) idxRight++; 
    else ende=1;
  }  

  stepsX=dx*STEPS_PER_MM; // Koordinatendifferenz in Motorschritte umrechnen 
  stepsY=dy*STEPS_PER_MM;

  // Vektorlänge in Schritten (steps) mittels Pythagoras berechnen:
  linDist = int(sqrt(stepsX*stepsX + stepsY*stepsY)); 

  // Je größer die Tiefe, desto kleiner der zeitliche Abstand zwischen zwei Schritten (= Zahl der Clockticks). 
  // Der SpeedFactor ist für x und y auf einer Seite der Tragfläche immer gleich, 
  // aber er unterscheidet sich bei trapezförmigen Flächen zwischen rechter und linker Seite. 
  speedFactor = mittlereTiefe / tiefe;    

  // Berechnet, nach jeweils wievielen Clockticks ein Motorschritt gemacht wird. 
  // Je größer diese Werte sind, desto besser wird die Auflösung der Steigung. Ideal wäre eine sehr große Zahl. 
  // Das erhöht aber die Anzahl der Masterclockperioden pro Schritt und somit wird der Motor langsamer. 
  // Bei 20 ergibt sich ein guter Kompromiss zwischen Geschwindigkeit und Auflösung. Die Zahlen sind empirisch ermittelt.
  if (stepsX!=0) aspectX=speedFactor*linDist*20/stepsX; 
  else aspectX=linDist*100;    
  if (stepsY!=0) aspectY=speedFactor*linDist*20/stepsY; 
  else aspectY=linDist*100;

  // Parameter runden:
  aspX = int(aspectX);
  aspY = int(aspectY);
  stp = int(stepsX);

  println(ende+" # "+idxLeft+"-"+plx.length+" # "+y+" * "+aspX+" * "+aspY+" * "+stp);

  // Daten für die Übertragung aufbereiten -- Vorzeichen herausziehen, Betrag in Hex wandeln:
  if (stp>=0) stpSign='+'; 
  else stpSign='-';
  if (aspX>=0) aspXsign='+'; 
  else aspXsign='-';
  if (aspY>=0) aspYsign='+'; 
  else aspYsign='-';
  stpHex = hex(abs(stp));
  aspXhex = hex(abs(aspX));
  aspYhex = hex(abs(aspY));

  outStr = stpSign + stpHex.substring(4) +  aspXsign + aspXhex.substring(4) + aspYsign + aspYhex.substring(4);
  println(outStr);

  if (ende==0) return(outStr); 
  else return("#----#----#----"); // Abbruchbedingung prüfen
}

// Verfährt beide Seiten um distance in mm

void drive(char axis, int distance) { 
  String outStr, distHex;
  char sign;

  if (distance>0) sign = '+'; 
  else sign = '-';
  distHex = hex(abs(distance*STEPS_PER_MM));
  outStr = axis + sign + distHex.substring(4);
  println("Drive: " + outStr);
  myPort.write(outStr);
}

// Berechnet Schneidekoordinaten aus Rohkoordinaten

void applySettings() {
  int i;
  float tiefeL, tiefeR;
  float alphaOrg, alphaNew, deltaAlpha, len;
  float alphaSweep;
  float segmentLength, scaleL, scaleR;
  float d, dir, x1, x2, y1, y2;

  int lastx = 0, lasty = 0; 

  plx = new float [plxRaw.length];
  ply = new float [plyRaw.length];
  prx = new float [prxRaw.length];
  pry = new float [pryRaw.length];

  //=================== gewünschte Tiefe verrechnen ===================

  tiefeL = float(TLinks.getText()); // gewünschte Tiefe aus Textfeld lesen
  for (i=0; i<plxRaw.length; i++) {
    plx[i] = (1-plxRaw[i])*tiefeL; 
    ply[i] = plyRaw[i]*tiefeL;
  }     
  tiefeR = float(TRechts.getText()); // rechte Seite analog zur linken
  for (i=0; i<prxRaw.length; i++) {
    prx[i]=(1-prxRaw[i])*tiefeR;
    pry[i]=pryRaw[i]*tiefeR;
  }     

  //========== mittlere Tiefe, um später die relativen Geschwindigkeiten zu berechnen ==========

  mittlereTiefe = (tiefeL + tiefeR) /2;

  //================= Anstellwinkel verrechnen =================

  deltaAlpha = float(AWLinks.getText()); // gewünschten Winkel in Grad aus Textfeld lesen
  for (i=0; i<plx.length; i++) {   
    len = sqrt(sq(plx[i]) + sq(ply[i]));   // Abstand von der Endleiste zum aktuellen Koordinatenpaar
    // Koordinaten vom kartesischen System in Polarkoordinaten wandeln, dabei Division durch Null vermeiden:
    if (len!=0) alphaOrg = asin(ply[i]/len) / PI * 180; 
    else alphaOrg = 0;   
    alphaNew = alphaOrg + deltaAlpha; // Winkel addieren
    // zurückverwandeln in kartesische Koordinaten:
    plx[i] = len * cos(alphaNew / 180 * PI); 
    ply[i] = len * sin(alphaNew / 180 * PI); 
  }  

  deltaAlpha = float(AWRechts.getText()); // rechte Seite analog zur linken
  for (i=0; i<prx.length; i++) {
    len = sqrt(sq(prx[i]) + sq(pry[i]));
    if (len!=0) alphaOrg = asin(pry[i]/len) / PI * 180; 
    else alphaOrg=0;
    alphaNew = alphaOrg + deltaAlpha; 
    prx[i] = len * cos(alphaNew / 180 * PI);
    pry[i] = len * sin(alphaNew / 180 * PI); 
  }   

  //============ Pfeilungswinkel verrechnen (= Stauchung in x-Richtung) ============
  alphaSweep = float(Pfeilung.getText()); // gewünschte Pfeilung in Grad aus Textfeld lesen
  for (i=0; i<plx.length; i++) {
    plx[i]= plx[i] * cos(alphaSweep / 180 * PI);
  }  
  for (i=0; i<prx.length; i++) { // rechte Seite analog zur linken
    prx[i]= prx[i] * cos(alphaSweep / 180 * PI);
  }  

  //========= Segmentlänge verrechnen, um die tatsächliche X/Y-Bewegung zu erhalten =========

  segmentLength = float(SegLaenge.getText()); // gewünschte Segmentlänge aus Textfeld lesen
  tiefeL0 = tiefeL + (MOTOR_DIST-segmentLength) * (tiefeL - tiefeR) / 2 / segmentLength;
  tiefeR0 = tiefeR + (MOTOR_DIST-segmentLength) * (tiefeR - tiefeL) / 2 / segmentLength;
  scaleL = tiefeL0/tiefeL;
  scaleR = tiefeR0/tiefeR;

  for (i=0; i<plx.length; i++) {
    plx[i] = plx[i] * scaleL;  
    ply[i] = ply[i] * scaleL; 
  }   
  for (i=0; i<prx.length; i++) {
    prx[i] = prx[i] * scaleR;
    pry[i] = pry[i] * scaleR;
  }   

  //=================================== Beplankung abziehen ===================================
  d = float(Beplankung.getText()); // gewünschte Beplankungsdicke aus Textfeld lesen
  dir = 0; 
  alphaNew=PI/2; 
  for (i=0; i<plx.length-1; i++) { // linke Seite
    x1=plx[i]; 
    x2=plx[i+1];
    y1=ply[i]; 
    y2=ply[i+1];
    alphaOrg = asin((y1-y2) / sqrt(sq(y1-y2)+sq(x1-x2)));
    // um 90 Grad drehen, um Senkrechte auf dem Segment zu erhalten:
    alphaNew = alphaOrg-PI/2;   
    // Fallunterscheidung, ob es sich um die Ober- oder Unterseite der Fläche handelt:
    if (x1-x2>=0) dir=1; 
    else dir=-1;
    // neue Koordinaten berechnen:
    plx[i] = plx[i] - d * cos(alphaNew);
    ply[i] = ply[i] - d * dir * sin(alphaNew);
  }
  // für den letzten Punkt extra ausführen, da es keinen nächsten Punkt gibt, 
  // anhand dessen man den Winkel berechnen könnte, statt dessen werden die 
  // Winkel aus dem vorigen Schritt noch einmal benutzt 
  i = plx.length-1;
  plx[i] = plx[i] - d * cos(alphaNew);    
  ply[i] = ply[i] - d * dir * sin(alphaNew);

  for (i=0; i<prx.length-1; i++) { // rechte Seite analog zur linken
    x1=prx[i]; 
    x2=prx[i+1];
    y1=pry[i]; 
    y2=pry[i+1];
    alphaOrg = asin((y1-y2) / sqrt(sq(y1-y2)+sq(x1-x2)));
    alphaNew = alphaOrg-PI/2; 
    if (x1-x2>=0) dir=1; 
    else dir=-1;
    prx[i] = prx[i] - d * cos(alphaNew);
    pry[i] = pry[i] - d * dir * sin(alphaNew);
  }
  i = prx.length-1;
  prx[i] = prx[i] - d * cos(alphaNew);  
  pry[i] = pry[i] - d * dir * sin(alphaNew);
}

// Fährt den Schneidedraht in HOME-Position.
// Horizontal wird der Draht genau über den Winkelanschlag gefahren d.h. an die Stirnseite des Styroporblocks.
// Vertikal fährt der Draht auf die Position, die im GUI als Y-Startposition angegeben wird.
public void homeWire() {
  int xr, xl, yr, yl;
  int xDist, yDist;
  char inByte;
  String hexNumber;

  xr=0; 
  xl=0;
  myPort.write('8');    // Die beiden x-Achsen werden rückwärts bewegt.
  myPort.write('4');
  do {
    if (myPort.available() > 0) { 
      inByte=myPort.readChar();
      if (inByte=='A') xr=1;
      if (inByte=='B') xl=1;
    }
  } 
  while (xr*xl==0);   // Ende, wenn beide Seiten am Anschlag sind
  yr=0; 
  yl=0;
  myPort.write('6');    // Die beiden y-Achsen werden nach unten bewegt.
  myPort.write('2');
  do {
    if (myPort.available() > 0) { 
      inByte=myPort.readChar();
      if (inByte=='a') yr=1;
      if (inByte=='b') yl=1;
    }
  } 
  while (yr*yl==0);    // Ende, wenn beide Seiten am Anschlag sind

  yDist = int(yStart.getText());      // Draht auf die y-Startposition fahren
  hexNumber=hex((yDist-Y_STOP)*STEPS_PER_MM);   
  println("Schritte fahren: "+'U'+hexNumber.substring(4));
  myPort.write('U'+hexNumber.substring(4));

  hexNumber=hex(X_STOP*STEPS_PER_MM);  // Draht an den Anfang des Styroblocks fahren 
  println("Schritte fahren: "+'F'+hexNumber.substring(4));
  myPort.write('F'+hexNumber.substring(4));
}

// Verarbeitung der GUI-Events

public void controlEvent(ControlEvent theEvent) {
  int i;
  String profil;
  String hexNumber;

  if (theEvent.isGroup()) {
    println(theEvent.group().value()+" from "+theEvent.group());
    i = int(theEvent.group().value());
    profil=fileList[i];
    if (theEvent.group().name()=="Profil Links") loadFoil('L', profil);
    if (theEvent.group().name()=="Profil Rechts") loadFoil('R', profil);
  } 
  else 
  {  
    println("got a control event from controller with id "+theEvent.controller().id());
    switch(theEvent.controller().id()) {

      case(8): // Home
      println("Home");
      homeWire();
      break;

      case(9): // Start
      println("Start");
      applySettings();
      hexNumber = hex(Heizleistung); 
      myPort.write('H'+hexNumber.substring(6));   // Heizdraht einschalten
      delay(500);  // Warten bis aufgeheizt
      myPort.write('S');   // Der Arduino wird gestartet und holt sich dann selbst immer 
      // wieder neue Vektoren ab (siehe Funktion "draw")  
      break;

      case(10): // Cancel
      println("Cancel");
      myPort.write('0');
      break;

      case(11): // Exit
      println("Exit");
      exit();
      break;

      case(12): // Start Heizung
      hexNumber = hex(Heizleistung); 
      println("Heizung an:"+'H'+hexNumber.substring(6));
      myPort.write('H'+hexNumber.substring(6));
      break;

      case(21):
      println("UL");
      myPort.write('5');
      break;

      case(22):
      println("FL");
      myPort.write('7');
      break;

      case(23):
      println("BL");
      myPort.write('8');
      break;

      case(24):
      println("DL");
      myPort.write('6');
      break;

      case(31):
      println("UR");
      myPort.write('1');
      break;

      case(32):
      println("FR");
      myPort.write('3');
      break;

      case(33):
      println("BR");
      myPort.write('4');
      break;

      case(34):
      println("DR");
      myPort.write('2');
      break;
    }
  }
}

// Liest Daten des ausgewählten Profils und stellt es grafisch dar

public void loadFoil(char id, String profil) {
  int i;
  float x, y, lastx=0, lasty=0;
  String[] pieces; // Array, das die Zeilen der Datei enthält

  if (id=='L') { // linke Seite
    profilLinks = loadStrings(sketchPath+"/profile_dat/"+profil);
    plxRaw = new float [profilLinks.length]; // Array für die X-Koordinaten
    plyRaw = new float [profilLinks.length]; // Array für die Y-Koordinaten

    fill(100);
    stroke(60);
    rect(10, 10, 280, 230);
    ProfNameL.setValue(profil); // fügt Dateinamen in die Ausklappliste der GUI hinzu

    // bei der zweiten Zeile der Datei beginnen, die erste enthält den Namen des Profils:
    for (i=1; i<profilLinks.length; i++) { 
      pieces = splitTokens(profilLinks[i], " ");  
      x = float(pieces[0]);
      y = float(pieces[1]);
      plxRaw[i-1] = x;
      plyRaw[i-1] = y;
      // zeichnet den nächsten Abschnitt des Profils und
      // skaliert dazu die Koordinaten auf eine Tiefe von 250:
      stroke(255);
      if (i==1) {
        lastx=x*250; 
        lasty=y*250;
      };
      line(30+lastx, 220-lasty, 30+x*250, 220-y*250);
      lastx=x*250; 
      lasty=y*250;
    }
  }  

  if (id=='R') { // rechte Seite analog zur linken
    profilRechts = loadStrings(sketchPath+"/profile_dat/"+profil);
    prxRaw = new float [profilRechts.length];
    pryRaw = new float [profilRechts.length];

    fill(100);
    stroke(60);
    rect(300, 10, 280, 230); 
    ProfNameR.setValue(profil);

    for (i=1; i<profilRechts.length; i++) {
      pieces = splitTokens(profilRechts[i], " ");  
      x = float(pieces[0]);
      y = float(pieces[1]);
      prxRaw[i-1] = x;
      pryRaw[i-1] = y;
      stroke(255);
      if (i==1) {
        lastx=x*250; 
        lasty=y*250;
      };
      line(320+lastx, 220-lasty, 320+x*250, 220-y*250);
      lastx=x*250; 
      lasty=y*250;
    }
  }
  // setzt die Indizes für die aktuellen Koordinaten auf die Ausgangsposition 1,
  // damit der erste Eintrag übersprungen wird -- der ist immer 1.0 0.0
  idxLeft=1;
  idxRight=1;
  // setzt die Koordinaten für die vorige Position allesamt auf 0: 
  lastXR=0; 
  lastYR=0; 
  lastXL=0; 
  lastYL=0;
  println(pry);
}  

